@extends('layouts.main')

@section('content')
<div class="container-fluid">
    <div class="d-flex justify-content-between mb-2 align-items-center">
        <h6 class="fw-bold small">Stock Entry & Summary</h6>
        <input type="date" id="stockDate" value="{{ date('Y-m-d') }}" class="form-control form-control-sm w-auto">
    </div>

    <div class="col-md-4">
        <label for="showroomSelect">Select Showroom:</label>
        <select id="showroomSelect" class="form-control">
            <option value="">All Showrooms</option>
            @foreach($showrooms as $showroom)
                <option value="{{ $showroom->id }}">{{ $showroom->name }}</option>
            @endforeach
        </select>
    </div>

    <table class="table table-sm table-bordered text-end mb-2" id="summaryTable">
        <thead class="table-light">
            <tr>
                <th>OS ৳</th><th>Received ৳</th><th>R/D ৳</th><th>CS ৳</th><th>Sale ৳</th>
            </tr>
        </thead>
        <tbody><tr><td colspan="5">Loading...</td></tr></tbody>
    </table>

    <div class="d-flex justify-content-between">
        <div id="stockTableWrapper" style="width: 60%;">
            <h6 class="text-muted small mb-1">Stock Quantity</h6>
            <div class="accordion" id="categoryAccordion">
                </div>
        </div>

        <div id="categorySummaryWrapper" style="width: 39%;">
            <h6 class="text-muted small mb-1">Category-wise Money Summary</h6>
            <div id="categorySummaryContainer"></div>
        </div>
    </div>

    <div class="text-end mt-2">
        <button class="btn btn-sm btn-primary" id="saveBtn">Save</button>
    </div>
</div>
@endsection

@push('scripts')
<style>
    table {
        font-size: 11px;
    }
    td, th {
        padding: 2px !important;
        vertical-align: middle;
    }
    input[type="number"]::-webkit-inner-spin-button,
    input[type="number"]::-webkit-outer-spin-button {
        -webkit-appearance: none;
        margin: 0;
    }
    input[type="number"] {
        appearance: textfield;
        -moz-appearance: textfield;
    }
    input.form-control-xs {
        padding: 1px 4px;
        height: 24px;
        font-size: 11px;
        line-height: 1.2;
    }
    .accordion-button:not(.collapsed) {
        background-color: #f8f9fa;
    }
    .accordion-button {
        padding: 0.25rem 0.5rem;
        font-size: 11px;
    }
    .accordion-body {
        padding: 0;
    }
</style>


<script>
let stockData = {}, currentDate = document.getElementById("stockDate").value;

document.addEventListener('DOMContentLoaded', () => {
    // Initial fetch on page load
    fetchData(currentDate);

    // Event listener for date change
    document.getElementById("stockDate").addEventListener("change", e => {
        currentDate = e.target.value; // Update global currentDate
        fetchData(currentDate);
    });

    // Event listener for showroom selection change (NEW)
    document.getElementById("showroomSelect").addEventListener("change", () => {
        fetchData(currentDate); // Re-fetch data for the current date and new showroom
    });

    // Event listener for save button
    document.getElementById("saveBtn").addEventListener("click", saveData);
});

// fetchData function (Minimal changes)
function fetchData(date) {
    const showroomId = document.getElementById("showroomSelect").value; // Get showroomId here
    let url = `{{ route('stocks.fetch') }}?date=${date}`;
    if (showroomId) {
        url += `&showroom_id=${showroomId}`;
    }

    fetch(url)
        .then(res => {
            if (!res.ok) {
                return res.json().then(err => { throw err; });
            }
            return res.json();
        })
        .then(({ date, data, prevDate, selectedShowroomId }) => { // Get selectedShowroomId from response
            currentDate = date; // Keep this as it updates the global variable
            stockData = data;
            
            // Update the date input to reflect what was fetched (in case of default today)
            document.getElementById("stockDate").value = date;
            // Update the showroom select to reflect what was fetched (NEW)
            document.getElementById("showroomSelect").value = selectedShowroomId || '';

            renderTables();
            
            // Show previous date in console for debugging
            console.log(`Loaded data for ${date} (Showroom: ${selectedShowroomId || 'All'}). Previous date: ${prevDate}`);
        })
        .catch(error => {
            console.error('Error fetching stock data:', error);
            alert('Failed to load stock data. Check console for details.');
            document.getElementById('categoryAccordion').innerHTML = '<p class="text-danger p-3">Failed to load stock data.</p>';
            document.querySelector("#summaryTable tbody").innerHTML = '<tr><td colspan="5">Error loading data.</td></tr>';
            document.getElementById("categorySummaryContainer").innerHTML = '';
        });
}

function renderTables() {
    const accordion = document.getElementById('categoryAccordion');
    accordion.innerHTML = '';

    Object.entries(stockData).forEach(([cat, items], index) => {
        const accordionItem = document.createElement('div');
        accordionItem.className = 'accordion-item';
        accordionItem.innerHTML = `
            <h2 class="accordion-header" id="heading${index}">
                <button class="accordion-button collapsed" type="button" 
                        data-bs-toggle="collapse" data-bs-target="#collapse${index}" 
                        aria-expanded="false" aria-controls="collapse${index}">
                    ${cat}
                </button>
            </h2>
            <div id="collapse${index}" class="accordion-collapse collapse" 
                 aria-labelledby="heading${index}" data-bs-parent="#categoryAccordion">
                <div class="accordion-body p-0">
                    <table class="table table-sm mb-0">
                        <thead>
                            <tr>
                                <th>Product</th>
                                <th>Price</th>
                                <th>OS</th>
                                <th>Received</th>
                                <th>R/D</th>
                                <th>CS</th>
                                <th>Sale</th>
                            </tr>
                        </thead>
                        <tbody id="tbody${index}"></tbody>
                    </table>
                </div>
            </div>
        `;
        accordion.appendChild(accordionItem);

        const tbody = document.getElementById(`tbody${index}`);
        items.forEach((item) => {
            // Note: The 'os' input is only readonly if there was existing stock
            const osInput = (item.has_stock || item.has_prev_stock)
                ? `<input type="number" value="${item.os}" class="form-control form-control-xs" data-col="os" readonly style="background-color:#e9ecef;">`
                : `<input type="number" value="${item.os}" class="form-control form-control-xs" data-col="os">`;

            const row = document.createElement('tr');
            row.dataset.id = item.id;
            row.dataset.cat = cat;
            row.dataset.price = item.price;
            row.innerHTML = `
                <td>${item.name}</td>
                <td>${item.price.toFixed(2)}</td>
                <td>${osInput}</td>
                <td><input type="number" value="${item.received}" class="form-control form-control-xs" data-col="received"></td>
                <td><input type="number" value="${item.r_d}" class="form-control form-control-xs" data-col="r_d"></td>
                <td><input type="number" value="${item.cs}" class="form-control form-control-xs" data-col="cs"></td>
                <td><input type="number" value="${item.sale}" class="form-control form-control-xs" disabled style="background-color:#f8f9fa;"></td>
            `;
            tbody.appendChild(row);
        });
    });

    if (typeof bootstrap !== 'undefined') {
        new bootstrap.Collapse(document.getElementById('categoryAccordion'), {
            toggle: false
        });
    }

    attachListeners();
    updateCategorySummary();
    updateSummaryTable();
}

// Update these functions to work with the new structure
function attachListeners() {
    // It's crucial to remove previous listeners to prevent duplicates
    // after DOM updates, which can happen if renderTables is called multiple times.
    document.querySelectorAll('.accordion-body input[data-col]').forEach(input => {
        input.removeEventListener("input", updateRowAndSummaries); // Remove existing listener
        input.addEventListener("input", updateRowAndSummaries); // Add new listener
    });
}

// Extracted for clarity and reusability, but keeps original logic
function updateRowAndSummaries() {
    const row = this.closest("tr");
    const price = parseFloat(row.dataset.price || 0);
    const os = +row.querySelector("input[data-col='os']").value || 0;
    const r = +row.querySelector("input[data-col='received']").value || 0;
    const rd = +row.querySelector("input[data-col='r_d']").value || 0;
    const cs = +row.querySelector("input[data-col='cs']").value || 0;
    const sale = os + r - rd - cs;
    
    row.querySelector("input[disabled]").value = sale;
    
    updateCategorySummary();
    updateSummaryTable();
}


function updateSummaryTable() {
    let sum = {os:0, received:0, r_d:0, cs:0, sale:0};
    
    // Find all rows in all accordion bodies
    document.querySelectorAll('.accordion-body tbody tr[data-id]').forEach(row => {
        const price = parseFloat(row.dataset.price || 0);
        const os = +row.querySelector("input[data-col='os']").value || 0;
        const r = +row.querySelector("input[data-col='received']").value || 0;
        const rd = +row.querySelector("input[data-col='r_d']").value || 0;
        const cs = +row.querySelector("input[data-col='cs']").value || 0;
        const sale = os + r - rd - cs;
        
        sum.os += os * price;
        sum.received += r * price;
        sum.r_d += rd * price;
        sum.cs += cs * price;
        sum.sale += sale * price;
    });
    
    document.querySelector("#summaryTable tbody").innerHTML = `
        <tr>
            <td class="text-end">${sum.os.toFixed(2)}</td>
            <td class="text-end">${sum.received.toFixed(2)}</td>
            <td class="text-end">${sum.r_d.toFixed(2)}</td>
            <td class="text-end">${sum.cs.toFixed(2)}</td>
            <td class="text-end fw-bold">${sum.sale.toFixed(2)}</td>
        </tr>`;
}

function updateCategorySummary() {
    const catTotals = {};

    // Find all rows in all accordion bodies
    document.querySelectorAll('.accordion-body tbody tr[data-id]').forEach(row => {
        const cat = row.dataset.cat;
        const price = parseFloat(row.dataset.price || 0);
        const os = +row.querySelector("input[data-col='os']").value || 0;
        const r = +row.querySelector("input[data-col='received']").value || 0;
        const rd = +row.querySelector("input[data-col='r_d']").value || 0;
        const cs = +row.querySelector("input[data-col='cs']").value || 0;
        const sale = os + r - rd - cs;

        catTotals[cat] ??= { os: 0, received: 0, r_d: 0, cs: 0, sale: 0 };
        catTotals[cat].os += os * price;
        catTotals[cat].received += r * price;
        catTotals[cat].r_d += rd * price;
        catTotals[cat].cs += cs * price;
        catTotals[cat].sale += sale * price;
    });

    let html = '';
    Object.entries(catTotals).forEach(([cat, s]) => {
        html += `
        <div class="border p-2 mb-2 bg-light">
            <div class="fw-bold small mb-1">${cat}</div>
            <table class="table table-bordered table-sm text-end mb-0" style="font-size:11px;">
                <thead class="table-light">
                    <tr>
                        <th>OS ৳</th><th>Received ৳</th><th>R/D ৳</th><th>CS ৳</th><th>Sale ৳</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td>${s.os.toFixed(2)}</td>
                        <td>${s.received.toFixed(2)}</td>
                        <td>${s.r_d.toFixed(2)}</td>
                        <td>${s.cs.toFixed(2)}</td>
                        <td><strong>${s.sale.toFixed(2)}</strong></td>
                    </tr>
                </tbody>
            </table>
        </div>`;
    });

    document.getElementById("categorySummaryContainer").innerHTML = html;
}

// saveData function (Minimal changes)
function saveData() {
    const stockDate = document.getElementById("stockDate").value; // Get current date from input
    const showroomId = document.getElementById("showroomSelect").value; // Get showroomId here
    const payload = {};
    
    // Find all rows in all accordion bodies
    document.querySelectorAll('.accordion-body tbody tr[data-id]').forEach(row => {
        const id = row.dataset.id;
        payload[id] = {
            opening_stock: +row.querySelector("input[data-col='os']").value || 0,
            received: +row.querySelector("input[data-col='received']").value || 0,
            return_damage: +row.querySelector("input[data-col='r_d']").value || 0,
            closing_stock: +row.querySelector("input[data-col='cs']").value || 0
        };
    });

    console.log('Saving data:', { stock_date: stockDate, showroom_id: showroomId, entries: payload }); // Debug log

    fetch("{{ route('stocks.update') }}", {
        method: "POST",
        headers: {
            'X-CSRF-TOKEN': "{{ csrf_token() }}",
            'Content-Type': "application/json",
            'Accept': 'application/json'
        },
        body: JSON.stringify({ 
            stock_date: stockDate, 
            showroom_id: showroomId, // Pass the selected showroom ID here
            entries: payload 
        })
    })
    .then(res => {
        if (!res.ok) {
            return res.json().then(err => { throw err; });
        }
        return res.json();
    })
    .then(data => {
        console.log('Save response:', data); // Debug log
        alert(data.message || 'Stock saved successfully');
        // Refresh the data after save with the latest selected date and showroom
        fetchData(document.getElementById("stockDate").value);
    })
    .catch(error => {
        console.error('Save error:', error); // Debug log
        alert(error.message || 'Failed to save stock. Check console for details.');
    });
}
</script>
@endpush